/*
 * Screen.java - Video Terminal v 0.6
 * Part of "Cocoa Pom 1"
 * Copyright (C) 2000,2001 VERHILLE Arnaud
 * gist@wanadoo.fr
 * http://www.chez.com/apple1/
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */

import com.apple.cocoa.foundation.*;
import com.apple.cocoa.application.*;
import java.io.*;
import java.awt.Rectangle;

/**
*  The Screen Class behaves like the TV interface & The TV itself
 *
 * @author     VERHILLE Arnaud
 * @created    21 octobre 2000
 * @modified   Joe Crobak 4 January 2003 
 */
public class Screen extends NSView {
	
    // Private variablcharTablees
    private int charTable[][];        // character table created from apple1.vid
    private int screenTbl[][];
    private int indexX, indexY;
    private NSColor bgColor, textColor;
    private int pixelSize;
    private boolean scanline;
    private int terminalSpeed; // set to 60 char/s in the constructor... apparently the genuine speed.
    private long lastTime = System.currentTimeMillis();
    private Pia6820 pia;
	
    private static Screen _screen;
    
    // Constructor
    public Screen(NSRect frame) {
        super(frame);
        _screen = this;

        charTable = new int [128][8]; //new int[0x80][8];
		screenTbl = new int[40][24];
       	
		//loadDefaults();
        loadCharTable();
        reset();
        
		pia = Pia6820.sharedPia6820();
		// 6502 at 1Mhz, refresh with real time every 50 ms	
		//M6502.sharedM6502().start();
    }
	
	// the screen is constructed when the NSWindow is loaded at the program launch.
    public static Screen sharedScreen() {
		
        return _screen;		
    }
    
	public void loadDefaults() {
		
		bgColor = NSColor.blackColor();
        textColor = NSColor.greenColor();
		pixelSize = 1;
		scanline = false;   
		terminalSpeed = 60;  // 60 char/s is the genuine speed
	}
	
	// Reset the Screen by Filling it with no characters
	public void reset() {
		indexX = indexY = 0;
		initScreenTbl();
		setNeedsDisplay(true);
    }
	
	
    //*************************************************
    //************** Assinger methods *****************
    //*************************************************
	public void setBgColor(NSColor bg) {
		bgColor = bg;
	}
	
	public void setTextColor(NSColor text) {
		textColor = text;
	}
	
	public void setPixelSize(int n) {
		pixelSize = n;
		MenuController.sharedMenuController().getEmuWindow().setContentSize(new NSSize(pixelSize*280.0f,pixelSize*192.0f));
	}
	
	public void setTerminalSpeed(int n) {
		terminalSpeed = n;
	}
	
	public void setScanline(boolean scanline) {
		this.scanline = scanline;
    }
	
	//*************************************************
    //************** Accessor methods *****************
    //*************************************************
	public NSColor getBgColor() {
		return bgColor;
	}
	
	public NSColor getTextColor() {
		return textColor;
	}
	
	public int getPixelSize() {
		return pixelSize;
	}
	
	public int getTerminalSpeed() {
		return terminalSpeed;
	}
	
	public boolean getScanline() {
		return scanline;
	}
	

    //*************************************************
    //************** Video Terminal *******************
    //*************************************************
	
    // Output the character associated with dsp to screen.
    public void outputDsp(int dsp) {

        switch (dsp) {
			
            case 0x00: // Nothing
            case 0x7F: // 127 Nothing (Dsp Register Mask)
                break;
                
            case 0x5F: // 95 Backspace
                if (indexX == 0) {
                    indexY--;
                    indexX = 39;
                }
                else {
                    indexX--;
                }
                screenTbl[indexX][indexY] = 0x00;
                break;
                
            case 0x0A: // 10 End of Line
            case 0x0D: // 13 End of Line
                indexX = 0;
                indexY++;
                break;
                
            default:  // Standard Char
                screenTbl[indexX++][indexY] = dsp;
                break;
        }
        if (indexX == 40) {
            indexX = 0;
            indexY++;
        }
        if (indexY == 24) {
            newLine();
            indexY--;
        }
        
        delayOutputToMatchTerminalSpeed(); 
		display();
    }
	
    //**************************************************
    //*************** Graphical Methods ****************
    //**************************************************
    
    
    /**
	 *  Synchronize the Terminal Output to terminalSpeed char/s. that is, make the terminal wait until another
	 *  key can be pushed.
     *  This method freezes the whole program when it waits (This is not really a problem)
     */
    private void delayOutputToMatchTerminalSpeed() {
		// Apple1 Terminal Output terminalSpeed Charac each second
		int sleepMillis = (int) ((1000 / terminalSpeed) - (System.currentTimeMillis() - lastTime));
		if (sleepMillis > 0) {
			try {
				Thread.sleep(sleepMillis);
			}
			catch (Exception e) {
				System.out.println(e);
			}
		}
		lastTime = System.currentTimeMillis();
    }
    
    
    public void drawRect(NSRect frame) {

		int i, j, xPosition, yPosition;
        
		// Charac Resolution 7x8 ; Matrix 5x7
		// Screen Resolution 280x192
        
        // fill the background with black
		NSRect myBounds = bounds();
        bgColor.set();
        NSBezierPath.fillRect(myBounds);
        
		textColor.set(); // set it to the text color to draw the characters
		for (i = 0; i < 40; i++) {
			for (j = 0; j < 24; j++) {
				xPosition = i * (pixelSize * 7);
				yPosition = j * (pixelSize * 8);
				drawChar(frame, xPosition, yPosition,screenTbl[i][j]);
			}
		}  
		// Cursor
		drawChar(frame, indexX * (pixelSize * 7), indexY * (pixelSize * 8), 0x01);

    }
    
    /**
		*  Draw a Character from the Character Table on Display
     *
     * @param  frame		 The NSRect on screen to draw on
     * @param  xPosition     The Upper Left X position
     * @param  yPosition     The Upper left Y position
     * @param  characNumber  The Character Value [0x00 -> 0xFF]
     */
    private void drawChar(NSRect frame, int xPosition, 
                          int yPosition, int characNumber) {
		int k, l;
        NSRect charArea;
        
		for (k = 0; k < 8; k++)
			for (l = 1; l < 8; l++)  // first bit unused (color in apple2)
				if ((charTable[characNumber][k] & (0x01 << l)) == (0x01 << l)) {
					charArea = new NSRect(xPosition + pixelSize * (l - 1), 
                                          yPosition + pixelSize * k, 
                                          pixelSize, 
                                          pixelSize - (scanline ? 1 : 0));
					NSBezierPath.fillRect(charArea);
				}
    }
		
	// cocoa needs this to know to handle key events			
	public boolean acceptsFirstResponder() {
			
		return true; 
	}
	
	// cocoa method makes the positive y-axis run downward	
	public boolean isFlipped() {
			
		return true; 		
	}
		
	public void keyDown(NSEvent theEvent) { 
			
		if (pia.getKbdInterrupts()) {
			int result = KeyboardTranslator.translateCharacter((theEvent.characters()).charAt(0));
			if (result != -1) {
				pia.writeKbd(result);
				pia.writeKbdCr(0xA7);
			} else {
				System.out.println("The key \"" + (theEvent.characters()).charAt(0) + "\" is an invalid key.");
			}
		} else {
			System.out.println("Keyboard interrupts disabled.");
		}
	}
	

	//******************************************************
	//******************** Utility methods *****************
	//******************************************************
	
	//Create the Character Table from apple1.vid
	private void loadCharTable() {

		NSBundle bund = NSBundle.mainBundle();
		String filename = bund.resourcePath() + "/apple1.vid";

		FileInputStream fis = null;
		try {
			fis = new FileInputStream(filename);
			int i,j,k=0,l;
			for (i = 0; i < 128; i++) {
				for (j = 0; j < 8; j++) {
					charTable[i][j] = fis.read();
				}
			}
			fis.close();
		}
		catch (Exception e) {
			System.out.println(e);
		}		
	}
	
	//  Fill the Screen Table with no Characters
	private void initScreenTbl() {
		
		int i, j;
		for (i = 0; i < 40; i++)
			for (j = 0; j < 24; j++)
				screenTbl[i][j] = 0;  // 0x00;		
	}
	
	//  Create a new line and scroll the old lines
	private void newLine() {
		
		int i, j;
		for (i = 0; i < 40; i++)
			for (j = 0; j < 23; j++)
				screenTbl[i][j] = screenTbl[i][j + 1];
		
		for (i = 0; i < 40; i++)
			screenTbl[i][23] = 0x00;
				
		setNeedsDisplay(true);
	}
		
}
