/* 
 * MenuController.java Cocoa v 0.1 
 * Part of "Cocoa Pom 1"
 * Copyright (c) 2004 Joe Crobak
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

import com.apple.cocoa.foundation.*;
import com.apple.cocoa.application.*;
import java.io.*;

/**
 *  The MenuController handles all of the callbacks from the MenuItems, preforms a memory load,
 *  and initiating the load and save of preferences.
 *
 * @author     Joe Crobak 
 * @created    19 June 2003
 * @modified   Joe Crobak 7 January 2004
 */
public class MenuController extends NSObject {

	private NSWindow mainWindow;
	private NSMenu menubar;
	private NSMenuItem scanline;
    private static MenuController _menuController;
    private NSOpenPanel op;
	private Screen screen;
	
    public MenuController() {
		super();      
		screen = Screen.sharedScreen();
    }
    
    public static MenuController sharedMenuController() {
    
        if (_menuController == null)
            _menuController = new MenuController();
            
        return _menuController;    
    }
	
	public void awakeFromNib() {
	
		System.out.println("loading preference file");
		if (MyPrefsController.sharedMyPrefsController().loadPreferencesFile() == false) {
		
			// load defaults everywhere
			Memory.sharedMemory().loadDefaults();
			Memory.sharedMemory().reset();
			Screen.sharedScreen().loadDefaults();
		}
		M6502.sharedM6502().start();
		NSNotificationCenter.defaultCenter().addObserver(this,
                                                        new NSSelector("applicationWillTerminate", new Class[] {NSNotification.class}),
                                                        "NSApplicationWillTerminateNotification",
                                                        null);
		Screen.sharedScreen().setNeedsDisplay(true);
		getEmuWindow().setFrameUsingName("cocoa pom 1"); // place the window at the location of the last run
		if (Screen.sharedScreen().getScanline() == true)
			scanline.setState(NSCell.OnState);
		else
			scanline.setState(NSCell.OffState);
	}
	
    public NSWindow getEmuWindow() {
	
		NSArray windows = NSApplication.sharedApplication().windows();
		for (int i=0; i<windows.count(); i++) {
			if (((NSWindow)(windows.objectAtIndex(i))).title().equals("Cocoa Pom 1"))
				return (NSWindow)(windows.objectAtIndex(i));
		}
		return null;
    }
	
    public void fileSave(Object sender) { /* IBAction */
        
		NSApplication.sharedApplication().beginSheet(MySaveSheetController.sharedMySaveSheetController().window(), // sheet
													getEmuWindow(), // doc window
													this,
													null,
													null);
    }

    public void fileLoad(Object sender) { /* IBAction */
    
		// not used as to allow all file extensions possible.
        NSMutableArray fileExts = new NSMutableArray();
		fileExts.addObject("txt");

		op = NSOpenPanel.openPanel();
        op.setAllowsMultipleSelection(false);
        op.beginSheetForDirectory(null, // directory
                                  null, // filename
                                  null,  // file types
                                  MenuController.sharedMenuController().getEmuWindow(), // document window
                                  this, // modalDelegate
                                  new NSSelector("performMemoryLoad::",new Class[] {getClass()}), // didendselector
                                  null); // contextInfo
    }
	
	public void performMemoryLoad(NSOpenPanel sheet, int returnCode, Object contextInfo) {
    
        FileInputStream fis = null;
        String fileName;
            
        if (returnCode == NSOpenPanel.OKButton) {
        
			fileName = (String)op.filenames().objectAtIndex(0);
			try {
				int i, k, address, value;
				String _address, _value, _strLine;
				fis = new FileInputStream(fileName);
				
				BufferedReader _br = new BufferedReader(new InputStreamReader(fis));
													 
				while ( true ) {	// empty string, last line
					
					_strLine = _br.readLine();
					if (_strLine == null) // last line, exit loop
						break;
					// skip blank line or comment
					if (_strLine.length() == 0 || _strLine.charAt(0) == '/')
						continue;
					
					// Get the address from the string
					for (k=0; k<_strLine.length(); k++)
						if (_strLine.charAt(k) == ':')
							break;
					_address = _strLine.substring(0, k);
					try {
						address = Integer.parseInt(_address.trim(), 16 /* radix */ );
					}
					catch (Exception e) { // invalid address
						//System.out.println(e);
						continue;
					}
					
					for (i = k+2; i < _strLine.length(); i += 3) {
					
						_value = _strLine.substring(i,i + 2);
						try {
							value= Integer.parseInt(_value.trim(), 16 /* radix */ );
							
							// If exception isn't thrown, then it's a good value so put it in Memory
							Memory.sharedMemory().write((address + ((i - (k+2)) / 3)), value);
						}
						catch (Exception e) { // invalid address
							//System.out.println(e);
							break;
						}
					}		
				}
			}
			catch (Exception e) { // likely a file i/o exception
				System.out.println(e);
			}
        } // NSOpenPanel.OKButton
        else if (returnCode == NSOpenPanel.CancelButton) {
        
           // System.out.println("NSOpenPanel.CancelButton");        
        }    
    }

    public void emuReset(Object sender) { /* IBAction */    
        
        Pia6820.sharedPia6820().reset();
        M6502.sharedM6502().reset();    
    }
		    
    public void emuHard(Object sender) { /* IBAction */
        
        M6502 micro = M6502.sharedM6502();
        micro.stop();
        micro.reset();
        
        Pia6820.sharedPia6820().reset();
        Screen.sharedScreen().reset();
        Memory.sharedMemory().reset();
        
        try {
            Thread.sleep(200);    // To avoid wrong synchronized methods
        }
        catch (Exception e) {
            System.out.println(e);
        }
        micro.start();
    }
	
	public void emuConfig(Object sender) { /* IBAction */
    
		MyPrefsController.sharedMyPrefsController().showWindow(sender);      
    }
	
	public void normalSize(Object sender) { /* IBAction */
		//System.out.println("normalSize");
		Screen.sharedScreen().setPixelSize(1);
		Screen.sharedScreen().setScanline(false);
		scanline.setState(NSCell.OffState);
		Screen.sharedScreen().setNeedsDisplay(true);
	}
	
	public void doubleSize(Object sender) { /* IBAction */
		//System.out.println("doubleSize");
		Screen.sharedScreen().setPixelSize(2);
		Screen.sharedScreen().setNeedsDisplay(true);
	}
	
	public void tripleSize(Object sender) { /* IBAction */
		//System.out.println("tripleSize");
		Screen.sharedScreen().setPixelSize(3);
		Screen.sharedScreen().setNeedsDisplay(true);
	}
	
	public void toggleScanline(Object sender) { /* IBAction */

		if (scanline.state() == NSCell.OffState) {
			scanline.setState(NSCell.OnState);
			Screen.sharedScreen().setScanline(true);
		} 
		else {
			scanline.setState(NSCell.OffState);
			Screen.sharedScreen().setScanline(false);
		}
		Screen.sharedScreen().setNeedsDisplay(true);
	}
	
	// called for each menu item in the menu.  this returns false if the screen size is 1 and the menu item
	// is the scanline menuitem, because in that case we don't want to give the user the option of the scanline	
	public boolean validateMenuItem(Object sender) {
	
		if ((((NSMenuItem)sender).title()).equals("Scanline") && Screen.sharedScreen().getPixelSize() == 1)
			return false;
		return true;
	}
	
	// last thing called before application quits
	public void applicationWillTerminate(NSNotification aNotification) {

//		System.out.println("Saving preferences to file");
		MyPrefsController.sharedMyPrefsController().savePreferencesFile();
		getEmuWindow().saveFrameUsingName("cocoa pom 1"); // saves the window location in the user defaults
	}
}
