/*
 * Memory.java - Cocoa v 0.1 is a modified version of memory v 0.5 by:
 * Part of "Cocoa Pom 1"
 * Copyright (C) 2000 VERHILLE Arnaud
 * gist@wanadoo.fr
 * http://www.chez.com/apple1/
 * 
 * -- Modified/updated for Cocoa Pom by Joe Crobak (c) 2004--
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */

import java.io.*;
import com.apple.cocoa.foundation.*;
import com.apple.cocoa.application.*;

/**
 *  
 * This is a simple Memory int[0x10000] object.
 *
 * @author     VERHILLE Arnaud 
 * @created    21 octobre 2000 
 * @modified   Joe Crobak 4 January 2004
 */
public class Memory {
    private boolean ram8k;
    private boolean writeInRom;
    private boolean appletMode = false;
    private int[] mem; 
    private Pia6820 pia;
    private static Memory _mem;
    private int memorySize;
    /**
     *  constructor for an Executable. Initialise the Memory by filling it with zero 
     *
     * @param  pia  Pia Interface Object
     */
    public Memory() {
		
		pia = Pia6820.sharedPia6820();        
    }
    
    public static Memory sharedMemory(){
    
        if (_mem == null)
            _mem = new Memory();
        
        return _mem;
    }
    
	public void loadDefaults() {
		memorySize=0x10000; // pretty sure this means 64K
		mem = new int[memorySize]; 
		ram8k = false;
		writeInRom = true;
	}

	// returns in kilobytes
	public int getMemorySize() {
		return memorySize / 1024;
	}
	
	// passed an amount of memory in kilobytes
	public void setMemorySize(int size) {
		memorySize = size *1024;
		mem = new int[memorySize];
		if (size <= 8)
			setRam8k(true);
		else
			setRam8k(false);
	}

    /**
     *  Sets the Ram8k attribute of the Memory object 
     *
     * @param  b  The new Ram8k value 
     */
    public void setRam8k(boolean b) {
		ram8k = b;
    }    
    /**
     *  Sets the WriteInRom attribute of the Memory object 
     *
     * @param  b  The new WriteInRom value 
     */
    public void setWriteInRom(boolean b) {
		writeInRom = b;
    }
    /**
     *  Read a memory Address 
     *
     * @param  address  Range 0 to 0xFFFF 
     * @return          It must be a byte (0 - 0xFF) 
     */
    public int read(int address) {
		if (address == 0xD013) {
			// Video I/O DSPCR
			return pia.readDspCr();
		}
		if (address == 0xD012) {
			return pia.readDsp();
		}
		if (address == 0xD011) {
			// Keyboard I/O KBDCR
			return pia.readKbdCr();
		}
		if (address == 0xD010) {
			return pia.readKbd();
		}
		return mem[address];
    }
    /**
     *  Write a value in a memory address 
     *
     * @param  address  Range 0 to 0xFFFF 
     * @param  value    It must be a byte (0 - OxFF) 
     */
    public void write(int address, int value) {
		// I/O memory Address
		if (address == 0xD013) {
			// Video I/O DSPCR
			mem[address] = value;
			pia.writeDspCr(value);
			return;
		}
		if (address == 0xD012) {
			mem[address] = value;
			pia.writeDsp(value);
			return;
		}
		if (address == 0xD011) {
			// Keyboard I/O KBDCR
			mem[address] = value;
			pia.writeKbdCr(value);
			return;
		}
		if (address == 0xD010) {
			mem[address] = value;
			pia.writeKbd(value);
			return;
		}
		// RAM and ROM Address
		if (address >= 0xFF00) {
			// Do not write in Rom ?
			if (!writeInRom) {
			return;
			}
		}
		if (ram8k) {
			// only 8kb of Ram
			if (address >= 0x2000) {
				if (address < 0xFF00) {
					return;
				}
			}
		}
		mem[address] = value;
    }

    // **********************************************
    // ************ Utility Methods  ****************
    // **********************************************
    /**
     *  Reset the Memory <BR> Fill it with zero then load the bios
     */
    public void reset() {
		for (int i = 0; i < memorySize; i++) {
			mem[i] = 0x00;
		}
		loadRom();
    }
    /**
     *  Dump the Memory from start to end
     *
     * @param  start  int value [0x00, memorySize-1]
     * @param  end    int value [0x00, memorySize-1]
     * @return        Int[]
     */
    public int[] dumpMemory(int start, int end) {
		int i;
		int[] fbrut = new int[end - start + 1];
		for (i = 0; i < (end - start + 1); i++) {
			fbrut[i] = (mem[start + i] & 0xFF);
		}
		return fbrut;
    }
    /**
     *  Load the Wozniak Monitor BIOS in Memory "Use Executable or Applet Functions"
     */
    public void loadRom() {

		NSBundle bund = NSBundle.mainBundle();
		String filename = bund.resourcePath() + "/apple1.rom";
//        System.out.println("apple1.rom filepath: " + filename);
        FileInputStream fis = null;
		File file = new File(filename);

		int startingAddress = 0xFF00; // genuine apple 2 starting address
		try {
            fis = new FileInputStream(filename);
            int i;
            for (i = startingAddress; i < 0x10000; i++) {
                mem[i] = fis.read();
            }
            fis.close();
        }
        catch (Exception e) {
            System.out.println(e);
        } 
    }
}
